/* Filename: DiscourseRelation.java
 * Author: Mark A. Finlayson
 * Format: Java 2 v1.5.0
 * Date created: Oct 16, 2007
 */
package edu.mit.discourse.core.rep.relation;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import edu.mit.story.core.align.IAlignedStoryModel;
import edu.mit.story.core.desc.IData;
import edu.mit.story.core.desc.IDesc;
import edu.mit.story.core.mappers.BlockMapper;
import edu.mit.story.core.mappers.PropertyMapper;
import edu.mit.story.core.model.IStoryData;
import edu.mit.story.core.model.IStoryModel;
import edu.mit.story.core.position.IHasPosition;
import edu.mit.story.core.position.PositionUtils;
import edu.mit.story.core.property.IProperty;
import edu.mit.story.core.util.ArrayUtils;


/** One line description goes here...
 * More detail goes here...
 *
 * @author Mark A. Finlayson
 * @version 1.00, (Oct 16, 2007)
 * @since 1.5.0
 */
public class Relation implements IRelation {
	
	private final String text;
	private final ILexicalMarker lexMarker;
	private final List<IArgument> arguments;
	private final List<IProperty> properties;
		
	public Relation(ILexicalMarker lexMarker, List<? extends IArgument> args, IStoryData data){
		this(lexMarker, args, null, data);
	}
	
	public Relation(ILexicalMarker lexMarker, List<? extends IArgument> args, List<? extends IProperty> props, IStoryData data){
		if(lexMarker == null) throw new NullPointerException();
		if(args.size() < 2) throw new IllegalArgumentException();
		
		this.lexMarker = lexMarker;
		this.arguments = ArrayUtils.makeNewUnmodifiableList(args);
		this.properties = ArrayUtils.makeNewUnmodifiableList(props);
		this.text = generateText(data);
	}
	
	private Relation(String text, ILexicalMarker lexMarker, List<IArgument> args, List<IProperty> props){
		this.text = text;
		this.lexMarker = lexMarker;
		this.arguments = args;
		this.properties = props;
	}
	
	public Relation clone(){
		return new Relation(text, lexMarker, arguments, properties);
	}
	
	/* (non-Javadoc) @see edu.mit.story.core.util.IHasText#updateText(edu.mit.story.core.provider.IProviderData) */
	protected String generateText(IStoryData data) {
		StringBuilder sb = new StringBuilder();
		if(lexMarker.isNull()){
			sb.append(getArguments().get(0).getDisplayText());
			sb.append(" ");
		}
		sb.append(lexMarker.getDisplayText());
		return sb.toString();
	}
	
	public ILexicalMarker getLexicalMarker(){
		return lexMarker;
	}
	
	public List<IArgument> getArguments(){
		return arguments;
	}
	
	/* (non-Javadoc) @see edu.mit.story.core.property.IHasProperties#getProperties() */
	public List<IProperty> getProperties() {
		return properties;
	}
	
	/* (non-Javadoc) @see edu.mit.story.core.util.IHasText#getText() */
	public String getDisplayText() {
		return text;
	}
	
	/* 
	 * (non-Javadoc) @see edu.mit.story.core.desc.IStructuredData#getDisplayPosition()
	 */
	
	public IHasPosition getDisplayPosition() {
		return calculatePosition();
	}

	/* (non-Javadoc) @see edu.mit.story.core.desc.IStructuredData#calculatePosition() */
	public IHasPosition calculatePosition() {
		Set<IHasPosition> p = new HashSet<IHasPosition>(1 + arguments.size());
		if(lexMarker.isExplicit()) p.add(lexMarker.calculatePosition());
		for(IArgument arg : arguments) p.add(arg.calculatePosition());
		return PositionUtils.combineAll(p);
	}

	/* (non-Javadoc) @see edu.mit.story.core.desc.IStructuredData#refresh(edu.mit.story.core.provider.IProviderData) */
	public IRelation recalculate(IDesc container, IStoryModel model) {
		
		List<IArgument> newArgs = null;

		ILexicalMarker newLexMarker = (ILexicalMarker)lexMarker.recalculate(container, model);
		if(newLexMarker == null) return null;
		if(newLexMarker != lexMarker) newArgs = new ArrayList<IArgument>(arguments); 
		
		IArgument newArg;
		for(IArgument oldArg : arguments){
			newArg = (IArgument)oldArg.recalculate(container, model);
			if(newArg == oldArg) continue;
			if(newArgs == null) newArgs = new ArrayList<IArgument>(arguments);
			newArgs.set(newArgs.indexOf(oldArg), newArg);
		}
		
		if(newArgs == null) return this;

		return newArgs.size() < 2 || lexMarker == null ? null : new Relation(newLexMarker, newArgs, properties, model.getData()); 
	}
	
	public static String serialize(IRelation relation){
		List<String> blocks = new ArrayList<String>(2 + relation.getArguments().size());
		blocks.add(LexicalMarker.serialize(relation.getLexicalMarker()));
		for(IArgument argument : relation.getArguments()) blocks.add(Argument.serialize(argument));
		blocks.add(PropertyMapper.getInstance().serialize(relation.getProperties()));
		return BlockMapper.PIPE.serialize(blocks);
	}
	
	public static Relation reconstitute(String description, IStoryData data){
		
		List<String> blocks = BlockMapper.PIPE.reconstitute(description);
		if(blocks.size() < 4) return null;

		LexicalMarker lexMarker = null;
		List<Argument> arguments = null;
		List<IProperty> properties = null;
		
		String block;
		Argument argument;
		boolean first = true;
		for(Iterator<String> i = blocks.iterator(); i.hasNext(); ){
			block = i.next();
			if(first){
				lexMarker = LexicalMarker.reconstitute(block, data);
				first = false;
			} else if(i.hasNext()){
				if(arguments == null) arguments = new ArrayList<Argument>(blocks.size()-2);
				argument = Argument.reconstitute(block, data);
				if(argument != null) arguments.add(argument);
			} else {
				properties = PropertyMapper.getInstance().reconstitute(block);
			}
		}
		
		if(lexMarker == null || properties == null || arguments == null || arguments.size() < 2) return null; // ill-formed
		return new Relation(lexMarker, arguments, properties, data);
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.desc.IData#equals(edu.mit.story.core.desc.IData, edu.mit.story.core.align.IAlignedStoryModel)
	 */
	public boolean equals(IData tgtData, IAlignedStoryModel model) {
		throw new UnsupportedOperationException();
	}


}
