/* Filename: AddEntityChange.java
 * Author: Mark A. Finlayson
 * Format: Java 2 v1.5.0
 * Date created: Oct 23, 2007
 */
package edu.mit.discourse.core.rep.refexp.changes;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import edu.mit.discourse.core.rep.coref.Coref;
import edu.mit.discourse.core.rep.coref.CorefRep;
import edu.mit.discourse.core.rep.coref.ICoref;
import edu.mit.discourse.core.rep.refexp.RefExpRep;
import edu.mit.story.core.desc.Desc;
import edu.mit.story.core.desc.IDesc;
import edu.mit.story.core.desc.IDescSet;
import edu.mit.story.core.meta.MetaDesc;
import edu.mit.story.core.meta.origin.IOrigin;
import edu.mit.story.core.meta.origin.Origin;
import edu.mit.story.core.meta.origin.OriginMetaRep;
import edu.mit.story.core.meta.timing.ITiming;
import edu.mit.story.core.meta.timing.TimingMetaRep;
import edu.mit.story.core.model.IMutableStoryModel;
import edu.mit.story.core.model.change.AbstractModelChange;
import edu.mit.story.core.model.change.IModelChange;
import edu.mit.story.core.model.change.ReplaceDescriptionChange;
import edu.mit.story.core.model.change.StoryChangeEvent;


/** One line description goes here...
 * More detail goes here...
 *
 * @author Mark A. Finlayson
 * @version 1.00, (Oct 23, 2007)
 * @since 1.5.0
 */
public class AddRefExpChange extends AbstractModelChange {
	
	protected final long corefID;
	protected final Set<Long> refexpIDs;
	protected final ITiming timing;
	
	public AddRefExpChange(IDesc coref, IDesc refexp, ITiming timing){
		this(coref, Collections.singleton(refexp), timing);
	}
	
	public AddRefExpChange(IDesc coref, Collection<? extends IDesc> refexps, ITiming timing){
		super("Add Referring Expression");
		
		// check arguments
		if(!CorefRep.getInstance().isType(coref))
			throw new IllegalArgumentException();
		if(refexps.isEmpty()) 
			throw new IllegalArgumentException();
		
		// construct hidden list
		Set<Long> hidden = new HashSet<Long>(refexps.size());
		for(IDesc refexp : refexps)
			hidden.add(refexp.getID());
		
		this.corefID = coref.getID();
		this.refexpIDs = Collections.unmodifiableSet(hidden);
		this.timing = timing;
	}
	
	/* (non-Javadoc) @see edu.mit.story.core.provider.changes.AbstractSynchronizedProviderChange#synchronizedDoApply(java.lang.Object, edu.mit.story.core.provider.IDescriptionProvider) */
	@Override
	protected StoryChangeEvent doApply(Object source, IMutableStoryModel model) {
		
		// get entity description
		IDesc oldDesc = model.getData().getDescriptions(CorefRep.getInstance()).getDescription(corefID);
		if(oldDesc == null) 
			return null;
				
		// get the entity, check and see if it already has this segment
		// construct the new reference set 
		IDescSet refexps = model.getData().getDescriptions(RefExpRep.getInstance());
		ICoref oldData = (ICoref)oldDesc.getData();
		Set<IDesc> newRefExps = new HashSet<IDesc>(oldData.getReferentialExpressions().size() + refexpIDs.size());
		newRefExps.addAll(oldData.getReferentialExpressions());
		IDesc newRefExp;
		boolean changed = false;
		for(Long refexpID : refexpIDs){
			newRefExp = refexps.getDescription(refexpID);
			if(newRefExp != null)
				changed |= newRefExps.add(newRefExp);
		}
		
		if(!changed)
			return null;
		
		// add the new description
		ICoref newData = new Coref(oldData.getName(), newRefExps);
		IDesc newDesc = new Desc(oldDesc.getID(), CorefRep.getInstance(), newData.calculatePosition(), newData, oldDesc.getMetaData());
		newDesc.getMetaData().addDesc(new MetaDesc<IOrigin>(OriginMetaRep.getInstance(), newDesc, Origin.USER_SPECIFIED));
		if(timing != null) 
			newDesc.getMetaData().addDesc(new MetaDesc<ITiming>(TimingMetaRep.getInstance(), newDesc, timing));

		IModelChange change = new ReplaceDescriptionChange(getName(), oldDesc, newDesc);
		return change.apply(source, model);
	}

}
