/* Filename: FScoreCalculator.java
 * Author: Andrew B. Shapiro
 * Format: Java 2 v1.5.0
 * Date created: Jul 7, 2008
 */
package edu.mit.discourse.core.compare;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.Assert;

import edu.mit.discourse.core.rep.relation.IArgument;
import edu.mit.discourse.core.rep.relation.ILexicalMarker;
import edu.mit.discourse.core.rep.relation.IRelation;
import edu.mit.discourse.core.rep.relation.RelationRep;
import edu.mit.story.core.desc.IDesc;
import edu.mit.story.core.desc.IDescSet;
import edu.mit.story.core.model.StoryModel;
import edu.mit.story.core.position.IHasPosition;
import edu.mit.story.core.position.InStepIterator;
import edu.mit.story.core.rep.IRep;

/**
 * The <code>FScoreCalculator</code> organizes and assembles the
 * IScoredDiscourse data structure, utilizing the f-score statistical method for
 * scoring.
 * 
 * Add remaining features
 * (1) In {@link #scoreDescSets(IDescSet, IDescSet)} make sure to  sort the scoredDiscourse list
 * (2) Fix for mismatched arguments in {@link #scoreArguments(List, List)}
 * 
 * @author Andrew B. Shapiro
 * @since 1.5.0 (Jul 7, 2008)
 */
public class DiscourseFScoreCalculator implements IScoreCalculator {
	
	public static final String KEY_Type = "Type";
	public static final String KEY_Score = "Score";
	public static final String KEY_Lid = "L-ID";
	public static final String KEY_Rid = "R-ID";
	public static final String KEY_Common = "Common";
	public static final String KEY_LComponents = "L-Components";
	public static final String KEY_RComponents = "R-Components";
	public static final String KEY_LText = "L-Text";
	public static final String KEY_RText = "R-Text";
	
	public static final String VAL_LexicalMarker = "Lexical Marker";
	public static final String VAL_NullLexicalMarker = "Null";
	public static final String VAL_ImplicitLexicalMarker = "Implicit";
	public static final String VAL_NonExlicitLexicalMarker = "Non-Explicit";
	public static final String VAL_Argument = "Argument";
	public static final String VAL_Adjunt = "Adjunct";
	public static final String VAL_Segment = "Segment";
	
	private static DiscourseFScoreCalculator instance;
	
	public int tabs;

	protected DiscourseFScoreCalculator() {}

	public static DiscourseFScoreCalculator getInstance() {
		if (instance == null) instance = new DiscourseFScoreCalculator();
		return instance;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see edu.mit.discourse.core.compare.IScoreCalculator#scoreStories(edu.mit.story.core.model.StoryModel,
	 *      edu.mit.story.core.model.StoryModel)
	 */
	public IScoredElement scoreStories(StoryModel leftStory, StoryModel rightStory) {

		IDescSet leftRelations = leftStory.getData().getDescriptions(RelationRep.getInstance());
		IDescSet rightRelations = rightStory.getData().getDescriptions(RelationRep.getInstance());
		
		Map<String, String> ps = new HashMap<String, String>();
		List<IScoredElement> children = scoreDescSets(leftRelations, rightRelations);

		// weighted common
		double common = 0;
		for (IScoredElement child : children) common += child.getScore();
		ps.put(KEY_Common, Double.toString(common));

		// number of left components
		int leftSize = leftRelations.size();
		ps.put(KEY_LComponents, Long.toString(leftSize));

		// number of right components
		int rightSize = rightRelations.size();
		ps.put(KEY_RComponents, Long.toString(rightSize));

		// f-score
		double storyScore = f_score(common, leftSize, rightSize);

		return new ScoredElement(storyScore, ps, children);
	}

	/**
	 * This method determines the representation of the argument
	 * <code>IDesc</code>'s and delegates their scoring to the appropriate
	 * method.
	 * 
	 * @return A <code>ScoredElement</code> representing the comparison of the
	 *         two argument <code>IDesc</code>'s.
	 * 
	 */
	protected IScoredElement scoreDescs(IDesc leftDesc, IDesc rightDesc) {

		Assert.isLegal(!(leftDesc == null && rightDesc == null));
		if (leftDesc != null && rightDesc != null) {
			Assert.isLegal(leftDesc.getRep() == rightDesc.getRep());
		}

		/* Initialize Generic Properties */
		Map<String, String> properties = new HashMap<String, String>();

		if (leftDesc != null)
			properties.put(KEY_Lid, Long.toString(leftDesc.getID()));
		if (rightDesc != null)
			properties.put(KEY_Rid, Long.toString(rightDesc.getID()));

		IRep rep = (leftDesc != null ? leftDesc.getRep() : rightDesc.getRep());
		properties.put(KEY_Type, rep.getName());
		IScoredElement scoredPair;
//		if (rep == SegmentRep.getInstance()) {
//			scoredPair = scoreSegmentSets(leftDesc, rightDesc, properties);
//		} else 
		if (rep == RelationRep.getInstance()) {
			scoredPair = scoreRelations(leftDesc, rightDesc, properties);
		} else {
			scoredPair = null;
		}
		tabs++;
		String indent = "";
		for (int tab = 0; tab < tabs; tab++) {
			indent += "\t";
		}
//		Debug.out(indent + "Scoring " + properties.get(KEY_Type) + " Pair");
		indent += "\t";
//		String leftID = properties.get(KEY_Lid);
//		String rightID = properties.get(KEY_Rid);
//		Debug.out(indent + (leftID == null ? "?" : leftID) + " <=> " + (rightID == null ? "?" : rightID));
//		Debug.out(indent + Long.toString((scoredPair != null ? scoredPair.getScore() : 0.0)));
		tabs--;
		return scoredPair;
	}

	/**
	 * Populates a <code>ScoredElement</code> that represents the f-score
	 * analysis of the two argument <code>IDesc</code>'s.
	 * 
	 * <dt>Precondition:</dt>
	 * <dd>The representation returned by either <code>leftDesc.getRep()</code>
	 * or <code>rightDesc.getRep()</code> is an instance of
	 * <code>SegmentRep</code>.</dd>
	 * 
	 * 
	 * @param leftDesc
	 *            description of the "lefthand" relation
	 * @param rightDesc
	 *            description of the "righthand" relation
	 * @param properties
	 *            a map describing the properties of the comparison
	 * @return a <code>ScoredElement</code> that represents the f-score
	 *         analysis for the given relations
	 */
	protected IScoredElement scoreRelations(IDesc leftDesc, IDesc rightDesc, Map<String, String> properties) {
		double score = 0;
		List<IScoredElement> children = new LinkedList<IScoredElement>();

		IRelation leftRelation = (leftDesc != null ? (IRelation) leftDesc.getData() : null);
		IRelation rightRelation = (rightDesc != null ? (IRelation) rightDesc.getData() : null);

		if (leftDesc != null && rightDesc != null) {

			/* Score Lexical Marker */
			children.add(scoreLexicalMarkers(leftRelation.getLexicalMarker(), rightRelation.getLexicalMarker()));

			/* Score Arguments */
			List<IArgument> leftArgs = leftRelation.getArguments();
			List<IArgument> rightArgs = rightRelation.getArguments();

			children.addAll(scoreArguments(leftArgs, rightArgs));
			children = new ArrayList<IScoredElement>(children);

			int leftRelationSize = leftArgs.size() + 1;
			properties.put(KEY_LComponents, Long.toString(leftRelationSize));

			int rightRelationSize = rightArgs.size() + 1;
			properties.put("R-Copmonents", Long.toString(rightRelationSize));

			double commonRelationScore = 0;
			for (IScoredElement child : children)
				commonRelationScore += child.getScore();

			properties.put(KEY_Common, Double.toString(commonRelationScore));

			score = f_score(commonRelationScore, leftRelationSize, rightRelationSize);
		}
		return new ScoredElement(score, properties, children);
	}
	
	protected IScoredElement scoreLexicalMarkers(ILexicalMarker leftLM, ILexicalMarker rightLM){
		
		// type
		Map<String, String> props = new HashMap<String, String>();
		props.put(KEY_Type, VAL_LexicalMarker);
		
		// if both are null, they must match
		if(leftLM.isNull() && rightLM.isNull()) {
			Map<String, String> childProps = new HashMap<String, String>();
			childProps.put(KEY_Type, VAL_NullLexicalMarker);
			childProps.put(KEY_LText, ILexicalMarker.nullDisplay);
			childProps.put(KEY_RText, ILexicalMarker.nullDisplay);
			IScoredElement child = new ScoredElement(1.0, childProps);
			return new ScoredElement(1.0, props, Collections.singletonList(child));
		}
		
		// if both are implicit they match only if their implicit markers match
		if(leftLM.isImplicit() && rightLM.isImplicit()) {
			Map<String, String> childProps = new HashMap<String, String>();
			childProps.put(KEY_Type, VAL_ImplicitLexicalMarker);
			boolean match= leftLM.getImplicitMarker().equalsIgnoreCase(rightLM.getImplicitMarker());
			int score = match ? 1 : 0;
			childProps.put(KEY_LText, leftLM.getImplicitMarker() + ILexicalMarker.implicitDisplay);
			childProps.put(KEY_RText, rightLM.getImplicitMarker() + ILexicalMarker.implicitDisplay);
			IScoredElement child = new ScoredElement(score, childProps);
			return new ScoredElement(score, props, Collections.singletonList(child));
		}
		
		// if one is null and one is implicit, they get a score of zero
		if(leftLM.isImplicit() && rightLM.isNull()) {
			Map<String, String> childProps = new HashMap<String, String>();
			childProps.put(KEY_Type, VAL_NonExlicitLexicalMarker);
			childProps.put(KEY_LText, ILexicalMarker.nullDisplay);
			childProps.put(KEY_RText, rightLM.getImplicitMarker() + ILexicalMarker.implicitDisplay);
			IScoredElement child = new ScoredElement(0, childProps);
			return new ScoredElement(0, props, Collections.singletonList(child));
		}
		
		// if one is null and one is implicit, they get a score of zero
		if(leftLM.isNull() && rightLM.isImplicit()) {
			Map<String, String> childProps = new HashMap<String, String>();
			childProps.put(KEY_Type, VAL_NonExlicitLexicalMarker);
			childProps.put(KEY_LText, leftLM.getImplicitMarker() + ILexicalMarker.implicitDisplay);
			childProps.put(KEY_RText, ILexicalMarker.nullDisplay);
			IScoredElement child = new ScoredElement(0, childProps);
			return new ScoredElement(0, props, Collections.singletonList(child));
		}
		
		List<IScoredElement> children = new ArrayList<IScoredElement>();
//		List<IScoredElement> children = scoreDescSets(leftLM.getSegments(), rightLM.getSegments());

		// if either is null or implicit, and the other neither null nor implicit, just add as an upaired child
		if(!leftLM.isExplicit() || !rightLM.isExplicit()){
			Map<String, String> childProps = new HashMap<String, String>();
			if(leftLM.isNull()){
				childProps.put(KEY_Type, VAL_NullLexicalMarker);
				childProps.put(KEY_LText, ILexicalMarker.nullDisplay);
			}
			if(rightLM.isNull()){
				childProps.put(KEY_Type, VAL_NullLexicalMarker);
				childProps.put(KEY_RText, ILexicalMarker.nullDisplay);
			}
			if(leftLM.isImplicit()){
				childProps.put(KEY_Type, VAL_ImplicitLexicalMarker);
				childProps.put(KEY_LText, leftLM.getImplicitMarker() + ILexicalMarker.implicitDisplay);
			}
			if(rightLM.isImplicit()){
				childProps.put(KEY_Type, VAL_ImplicitLexicalMarker);
				childProps.put(KEY_RText, rightLM.getImplicitMarker() + ILexicalMarker.implicitDisplay);
			}
			children.add(new ScoredElement(0.0, childProps));
		}

		// number of common elements
		double commonLMSize = 0;
		for (IScoredElement child : children) commonLMSize += child.getScore();
		props.put(KEY_Common, Double.toString(commonLMSize));

		// number of left components
		int leftLMSize = leftLM.getSegments().size();
		props.put(KEY_LComponents, Long.toString(leftLMSize));

		// number of right components
		int rightLMSize = rightLM.getSegments().size();
		props.put(KEY_RComponents, Long.toString(rightLMSize));

		// final score
		double score = f_score(commonLMSize, leftLMSize, rightLMSize);
		
		// make score element object
		return new ScoredElement(score, props, children);
	}
	
	protected List<IScoredElement> scoreArguments(List<IArgument> leftArgs, List<IArgument> rightArgs){
		throw new UnsupportedOperationException();
//		List<IScoredElement> results = new LinkedList<IScoredElement>();
//		Map<String, String> argumentProperties = new HashMap<String, String>();
//		List<IScoredElement> argumentChildren;
//
//		IArgument leftArg = null, rightArg = null;
//		double argScore, weightedArgMembership;
//		int leftArgSize = 0, rightArgSize = 0;
//
//		int size = Math.max(leftArgs.size(), rightArgs.size());
//		for (int argNum = 0; argNum < size; argNum++) {
//			argumentProperties.clear();
//			argumentProperties.put(KEY_Type, VAL_Argument);
//			try {
//				leftArg = leftArgs.get(argNum);
//				rightArg = rightArgs.get(argNum);
//
//				argumentChildren = scoreDescSets(leftArg.getSegments(), rightArg.getSegments());
//
//				weightedArgMembership = 0;
//				for (IScoredElement child : argumentChildren)
//					weightedArgMembership += child.getScore();
//				argumentProperties.put(KEY_Common, Double.toString(weightedArgMembership));
//
//				leftArgSize = leftArg.getSegments().size();
//				argumentProperties.put(KEY_LComponents, Long.toString(leftArgSize));
//
//				rightArgSize = rightArg.getSegments().size();
//				argumentProperties.put(KEY_RComponents, Long.toString(rightArgSize));
//
//				argScore = f_score(weightedArgMembership, leftArgSize, rightArgSize);
//			} catch (ArrayIndexOutOfBoundsException e) {
//				e.printStackTrace();
//				argumentChildren = new LinkedList<IScoredElement>();
//				argScore = 0;
//				for (IDesc segment : (argNum > leftArgs.size() ? rightArgs : leftArgs).get(argNum).getSegments()) {
//					argumentChildren.add((argNum > leftArgs.size() ? scoreDescs(null, segment) : scoreDescs(segment, null)));
//				}
//				if (argNum > leftArgs.size()) {
//					leftArgSize = leftArg.getSegments().size();
//					argumentProperties.put(KEY_LComponents, Long.toString(leftArgSize));
//				} else {
//					leftArgSize = rightArg.getSegments().size();
//					argumentProperties.put(KEY_RComponents, Long.toString(rightArgSize));
//				}
//			}
//			results.add(new ScoredElement(argScore, argumentProperties, argumentChildren));
//		}
//		return results;
	}

	/**
	 * Populates a <code>ScoredElement</code> that represents the f-score
	 * analysis of the two argument <code>IDesc</code>'s.
	 * 
	 * <dt>Precondition:</dt>
	 * <dd>The representation returned by either <code>leftDesc.getRep()</code>
	 * or <code>rightDesc.getRep()</code> is an instance of
	 * <code>SegmentRep</code>.</dd>
	 * 
	 * @param leftDesc
	 *            description of the "lefthand" segment
	 * @param rightDesc
	 *            description of the "righthand" segment
	 * @param properties
	 *            a map describing the properties of the comparison
	 * @return a <code>ScoredElement</code> that represents the f-score
	 *         analysis for the given segment
	 */
//	@SuppressWarnings("unchecked")
	protected ScoredElement scoreSegmentSets(IDesc leftDesc, IDesc rightDesc, Map<String, String> properties) {
		throw new UnsupportedOperationException();
//		double score = 0;
//		ISegment leftSegment = (leftDesc == null ? null : (ISegment) leftDesc.getData());
//		ISegment rightSegment = (rightDesc == null ? null : (ISegment) rightDesc.getData());
//
//		if (leftSegment != null)
//			properties.put(KEY_LText, leftSegment.getDisplayText());
//		if (rightSegment != null)
//			properties.put(KEY_RText, rightSegment.getDisplayText());
//
//		if (leftDesc != null && rightDesc != null) {
//			int weightedMembership = 0;
//			List<List<IDesc>> match;
//			InStepIterator<IDesc> inStepItr = new InStepIterator<IDesc>(leftSegment.getDescs(), rightSegment.getDescs());
//			String leftToken, rightToken;
//			while (inStepItr.hasNext()) {
//				match = inStepItr.next();
//				leftToken = ((IToken) match.get(0).get(0).getData()).getToken();
//				rightToken = ((IToken) match.get(1).get(0).getData()).getToken();
//				if (rightToken.equals(leftToken)) {
//					weightedMembership++;
//				}
//			}
//			properties.put(KEY_Common, Long.toString(weightedMembership));
//
//			int leftComponentsSize = leftSegment.getDescs().size();
//			properties.put(KEY_LComponents, Long.toString(leftComponentsSize));
//
//			int rightComponentsSize = rightSegment.getDescs().size();
//			properties.put(KEY_RComponents, Long.toString(rightComponentsSize));
//
//			score = f_score(weightedMembership, leftComponentsSize, rightComponentsSize);
//		}
//		return new ScoredElement(score, properties);
	}

	/**
	 * This method pairs and scores two sets of descriptions.

	 * @param leftSet
	 *            the "lefthand" set of descriptions
	 * @param rightSet
	 *            the "righthand" set of descriptions
	 * @return a list of <code>List&lt;ScoredElement&gt;</code> assembled
	 *         according to f-score analysis
	 */
	@SuppressWarnings("unchecked")
	protected List<IScoredElement> scoreDescSets(IDescSet leftSet, IDescSet rightSet) {
		List<IScoredElement> scoredDiscourse = new LinkedList<IScoredElement>();
//		String type = (leftSet != null ? leftSet.getRep().getName() : rightSet.getRep().getName());
		
		// start debugging code
//		tabs++;
//		String indent = "";
//		for (int tab = 0; tab < tabs; tab++) {
//			indent += "\t";
//		}
//		Debug.out(indent + "Scoring " + type + " Set");
		// end debugging code

		/* Exact Matches */
		InStepIterator<IDesc> inStepItr = new InStepIterator<IDesc>(leftSet, rightSet);
		List<IDesc> leftUnmatched = new ArrayList<IDesc>();
		List<IDesc> rightUnmatched = new ArrayList<IDesc>();

		List<List<IDesc>> matched, unmatched;
		unmatched = inStepItr.getUnmatched();
		if (!unmatched.get(0).isEmpty())
			leftUnmatched.addAll(unmatched.get(0));
		if (!unmatched.get(1).isEmpty())
			rightUnmatched.addAll(unmatched.get(1));

		while (inStepItr.hasNext()) {
			matched = inStepItr.next();
			scoredDiscourse.add(scoreDescs(matched.get(0).get(0), matched.get(1).get(0)));
			unmatched = inStepItr.getUnmatched();
			// Add unmatched items to a list for later pairing.
			if (!unmatched.get(0).isEmpty())
				leftUnmatched.addAll(unmatched.get(0));
			if (!unmatched.get(1).isEmpty())
				rightUnmatched.addAll(unmatched.get(1));
		}
		scoredDiscourse.addAll(pairUnmatched(leftUnmatched, rightUnmatched));
		
//		tabs--;
		return scoredDiscourse;
	}

	/**
	 * Organizes and iterates through the unpaired descriptions looking for
	 * best matches. As best matches are found they are removed from the
	 * prospective pool.
	 * 
	 * @param leftUnmatched
	 *            contains the unmatched descriptions from the left description
	 *            set
	 * @param rightUnmatched
	 *            contains the unmatched descriptions from the right description
	 *            set
	 * @return a <code>List&lt;ScoredElement&gt;</code> containing the
	 *         partially paired and unpaired descriptions
	 */
	protected List<IScoredElement> pairUnmatched(List<IDesc> leftUnmatched, List<IDesc> rightUnmatched) {
		List<IScoredElement> elements = new LinkedList<IScoredElement>();

		/* Sort Unmatched for Partial Pairing */
		List<IDesc> leftByLength = mergesortByLength(new LinkedList<IDesc>(leftUnmatched));
		List<IDesc> rightByLength = mergesortByLength(new LinkedList<IDesc>(rightUnmatched));

		/* Align and Pair Partial Matches */
		boolean inLeftAnnotation;
		IDesc pairTo = null;
		List<IDesc> prospectives = null;
		List<IDesc> partialMatches = new LinkedList<IDesc>();

		double highestScore;
		IScoredElement currentBestPair = null;
		IDesc currentBestMatch = null;
		IScoredElement currentPair = null;

		while (!leftByLength.isEmpty() && !rightByLength.isEmpty()) {
			inLeftAnnotation = (leftByLength.get(0).getLength() > rightByLength.get(0).getLength() ? false : true);
			if (inLeftAnnotation) {
				pairTo = rightByLength.remove(0);
				prospectives = leftByLength;
			} else {
				pairTo = leftByLength.remove(0);
				prospectives = rightByLength;
			}
			// find prospectives which overlap with pairTo
			for (IDesc prospective : prospectives) {
				/* No-overlap condition... */
				if (prospective.getRightOffset() <= pairTo.getOffset() || pairTo.getRightOffset() <= prospective.getOffset()) {
					continue;
				}
				partialMatches.add(prospective);
			}
			if (partialMatches.size() == 0) {
				if (inLeftAnnotation) {
					elements.add(scoreDescs(null, pairTo));
				} else {
					elements.add(scoreDescs(pairTo, null));
				}
			} else if (partialMatches.size() == 1) {
				if (inLeftAnnotation) {
					elements.add(scoreDescs(partialMatches.get(0), pairTo));
					leftByLength.remove(partialMatches.get(0));
				} else {
					elements.add(scoreDescs(pairTo, partialMatches.get(0)));
					rightByLength.remove(partialMatches.get(0));
				}
			} else {
				/* Find best partial match */
				highestScore = 0.0;
				for (IDesc partialMatch : partialMatches) {
					if (inLeftAnnotation) {
						currentPair = scoreDescs(partialMatch, pairTo);
					} else {
						currentPair = scoreDescs(pairTo, partialMatch);
					}
					if (highestScore < currentPair.getScore()) {
						currentBestPair = null;
						currentBestPair = currentPair;
						currentBestMatch = partialMatch;
						highestScore = currentPair.getScore();
					}
				}
				elements.add(currentBestPair);
				(inLeftAnnotation ? leftByLength : rightByLength).remove(currentBestMatch);
			}
			partialMatches.clear();
			pairTo = null;
		}
		/* Add left-over Segments */
		for (IDesc leftUnpaired : leftByLength) {
			elements.add(scoreDescs(leftUnpaired, null));
		}
		for (IDesc rightUnpaired : rightByLength) {
			elements.add(scoreDescs(null, rightUnpaired));
		}

		return elements;
	}

	public static double f_score(double commonScore, double... scores) {
		if(commonScore == 0) return 0;
		// Use the multiplicative inverse for each score
		for (int i = 0; i < scores.length; i++) {
			if(scores[i] == 0) throw new IllegalArgumentException("Score cannot be zero if commmon score is non-zero");
			scores[i] = Math.pow(scores[i], -1);
		}
		return commonScore * geometric_avg(scores);
	}

	public static List<IDesc> merge(List<IDesc> left, List<IDesc> right) {
		List<IDesc> result = new LinkedList<IDesc>();
		while (left.size() > 0 && right.size() > 0) {
			IHasPosition l = left.get(0);
			IHasPosition r = right.get(0);
			if (l.getLength() <= r.getLength()) {
				result.add(right.remove(0));
			} else {
				result.add(left.remove(0));
			}
		}
		if (left.size() > 0) {
			result.addAll(left);
		}
		if (right.size() > 0) {
			result.addAll(right);
		}
		return result;
	}

	public static double geometric_avg(double... scores) {
		if (scores.length == 0) {
			return 0.0;
		}
		double geometric_sum = 1.0;
		for (int i = 0; i < scores.length; i++) {
			geometric_sum *= scores[i];
		}
		return Math.pow(geometric_sum, 1.0 / scores.length);
	}

	public static List<IDesc> mergesortByLength(List<IDesc> list) {
		List<IDesc> left = new LinkedList<IDesc>();
		List<IDesc> right = new LinkedList<IDesc>();
		List<IDesc> result = new LinkedList<IDesc>();
		if (list.size() <= 1) {
			return list;
		}
		int middle = list.size() / 2;
		for (int i = 0; i < list.size(); i++) {
			if (i < middle) {
				left.add(list.get(i));
			} else {
				right.add(list.get(i));
			}
		}
		left = mergesortByLength(left);
		right = mergesortByLength(right);
		result = merge(left, right);
		return result;
	}
}