/* Filename: Adjunct.java
 * Author: M. A. Finlayson
 * Format: Java 2 v1.5.0
 * Date created: Dec 6, 2007
 */
package edu.mit.discourse.core.rep.relation;

import java.util.Collection;
import java.util.Collections;
import java.util.List;

import edu.mit.parsing.core.rep.token.TokenRep;
import edu.mit.story.core.align.IAlignedStoryModel;
import edu.mit.story.core.desc.Data;
import edu.mit.story.core.desc.IData;
import edu.mit.story.core.desc.IDesc;
import edu.mit.story.core.desc.ISegment;
import edu.mit.story.core.desc.Segment;
import edu.mit.story.core.desc.SegmentMapper;
import edu.mit.story.core.mappers.IValueMapper;
import edu.mit.story.core.model.IStoryData;
import edu.mit.story.core.model.IStoryModel;
import edu.mit.story.core.position.IHasPosition;
import edu.mit.story.core.position.IHasPositionSet;
import edu.mit.story.core.position.ImmutableHasPositionSet;
import edu.mit.story.core.position.SimplePosition;


public class Adjunct implements IAdjunct {

	private transient String text;
	private final ImmutableHasPositionSet<ISegment> segments;
	
	/**
	 * Constructs a new adjunct out of the specified collection of segments. The
	 * collection of segments may be {@code null} or empty; this merely creates
	 * an invalid adjunct (see {@link #isValid()} that has no segments.
	 * 
	 * @since SW 1.0
	 */
	public Adjunct(Collection<? extends ISegment> segments) {
		this.segments = (segments == null) ? new ImmutableHasPositionSet<ISegment>(Collections.<ISegment>emptySet()) : new ImmutableHasPositionSet<ISegment>(segments);
	}
	
	/* 
	 *(non-Javadoc) 
	 *
	 * @see edu.mit.story.core.util.IHasDisplayText#getDisplayText() 
	 */
	public String getDisplayText() {
		if(text == null) text = generateText();
		return text;
	}
	
	protected String generateText(){
		return Segment.generateText(segments);
	}
	
	/* 
	 *(non-Javadoc) 
	 *
	 * @see edu.mit.discourse.core.rep.relation.IAdjunct#getSegments() 
	 */
	public IHasPositionSet<ISegment> getSegments() {
		return segments;
	}

	/* 
	 *(non-Javadoc) 
	 *
	 * @see edu.mit.story.core.position.IHasPosition#getLength() 
	 */
	public int getLength() {
		return segments.getLength();
	}

	/* 
	 *(non-Javadoc) 
	 *
	 * @see edu.mit.story.core.position.IHasPosition#getOffset() 
	 */
	public int getOffset() {
		return segments.getOffset();
	}

	/* 
	 *(non-Javadoc) 
	 *
	 * @see edu.mit.story.core.position.IHasPosition#getRightOffset() 
	 */
	public int getRightOffset() {
		return segments.getRightOffset();
	}
	
	/* 
	 *(non-Javadoc) 
	 *
	 * @see java.lang.Object#clone() 
	 */
	public Adjunct clone(){
		return new Adjunct(segments);
	}

	/* 
	 *(non-Javadoc) 
	 *
	 * @see edu.mit.story.core.desc.IStructuredData#calculatePosition() 
	 */
	public IHasPosition calculatePosition() {
		return new SimplePosition(segments);
	}

	/**
	 * Indicates whether this adjunct is <em>valid</em>, that is, actually
	 * refers to at least one segment.
	 * 
	 * @return {@code true} if the adjunct has non-empty set of segments;
	 *         {@code false} otherwise.
	 * @since SW 1.0
	 */
	public boolean isValid() {
		return !segments.isEmpty();
	}
	
	/* 
	 * (non-Javadoc) @see edu.mit.story.core.desc.IStructuredData#getDisplayPosition()
	 */
	public IHasPosition getDisplayPosition() {
		return calculatePosition();
	}

	/* (non-Javadoc) @see edu.mit.story.core.desc.IStructuredData#refresh(edu.mit.story.core.provider.IDescriptionProvider) */
	public Adjunct recalculate(IDesc container, IStoryModel provider) {
		throw new UnsupportedOperationException();
	}	
	
	
	public static String serialize(IAdjunct adjunct){
		return SegmentMapper.serializeStatic(adjunct.getSegments());
	}
	
	public static IAdjunct reconstitute(String description, IStoryData data){
		IValueMapper<ISegment> mapper = new SegmentMapper(TokenRep.getInstance(), data);
		List<ISegment> segments = mapper.reconstitute(description);
		return new Adjunct(segments);
	}


	/* 
	 * (non-Javadoc) @see edu.mit.story.core.desc.IData#equals(edu.mit.story.core.desc.IData, edu.mit.story.core.align.IAlignedStoryModel)
	 */
	public boolean equals(IData tgtData, IAlignedStoryModel model) {
		if(this == tgtData)
			return true;
		if(!IAdjunct.class.isAssignableFrom(tgtData.getClass()))
			return false;
		
		IAdjunct tgtAdj = (IAdjunct)tgtData;
		return Data.equals(segments, tgtAdj.getSegments(), model);
	}

}
