/* Filename: TimeCalculator.java
 * Creator: M.A. Finlayson
 * Format: Java 2 v1.6.0
 * Date created: Feb 9, 2010
 */
package nil.ucm.indications2.ui.agreement;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import nil.ucm.indications2.ui.agreement.TimeResult.TimeResultBuilder;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IProgressMonitor;

import edu.mit.story.core.desc.IDesc;
import edu.mit.story.core.measure.IResult;
import edu.mit.story.core.measure.StoryFileExtractor;
import edu.mit.story.core.meta.IMetaDataMap;
import edu.mit.story.core.meta.IMetaDesc;
import edu.mit.story.core.meta.timing.ITiming;
import edu.mit.story.core.meta.timing.TimingMetaRep;
import edu.mit.story.core.model.IStoryModel;
import edu.mit.story.core.model.StoryModelImporter;
import edu.mit.story.core.rep.IRep;

/** 
 * TODO: Write comment
 *
 * @author M.A. Finlayson
 * @version $Rev$, $LastChangedDate$
 * @since nil.ucm.indications2.ui 1.0.0
 */
public class TimeCalculator {
	
	private final IRep rep;
	
	public TimeCalculator(IRep rep){
		this.rep = rep;
	}
	
	/** 
	 * Default idle timeout is five minutes
	 *
	 * @since nil.ucm.indications2.ui 1.0.0
	 */
	private final long idleTimeout = 1000*60*5;
	
	public List<IResult> calculate(IResource r, IProgressMonitor pm){
		if(r == null) throw new NullPointerException();
		
		List<IResult> results = new LinkedList<IResult>();
		List<IFile> files = StoryFileExtractor.getInstance().getTargets(r, results);
		
		// do counting over files we found
		pm.beginTask("Calculating Counts", files.size());
		IStoryModel model;
		for(IFile file : files){
			pm.subTask("Processing " + file.getName());
			model = StoryModelImporter.extractModel(file, results);
			if(model != null) results.add(calculate(file, model));
			pm.worked(1);
			if(pm.isCanceled()) return results;
		}
		pm.done();

		return results;
	}
	
	
	/**
	 * TODO: Write comment
	 *
	 * @param file
	 * @param rep
	 * @param model
	 * @return
	 * @since nil.ucm.indications2.ui 1.0.0
	 */
	public ITimeResult calculate(IFile file, IStoryModel model){
		if(model == null) throw new NullPointerException();
		
		// first get all the timings
		List<ITiming> timings = extractTimings(rep, model);
		
		TimeResultBuilder builder = new TimeResultBuilder();
		builder.rep = rep;
		
		// convert them to durations, calculating centroid along the way
		List<IDuration> durations = new LinkedList<IDuration>();
		for(ITiming timing : timings){
			if(timing.getStart() != null && timing.getStop() != null){
				durations.add(new Duration(timing.getStart().getTime(), timing.getStop().getTime()));
			} else if (timing.getStart() != null){
				durations.add(new Duration(timing.getStart()));
			} else if (timing.getStop() != null){
				durations.add(new Duration(timing.getStop()));
			}
		}

		// sort durations, calculate total time
		durations = new ArrayList<IDuration>(durations);
		Collections.sort(durations);
		LinkedList<IDuration> periods = calculatePeriods(durations);
		
		// calculate total
		IDuration longestPeriod = periods.getFirst();
		for(IDuration period : periods){
			builder.total += period.getLength();
			if(period.getLength() > longestPeriod.getLength()) longestPeriod = period;
		}
		
		// start and end
		builder.centroid = longestPeriod.getStart();
		builder.start = periods.getFirst().getStart();
		builder.end = periods.getLast().getEnd();
		
		return builder.toResult(file);
	}
	
	/**
	 * TODO: Write comment
	 *
	 * @param durations
	 * @return
	 * @since nil.ucm.indications2.ui 1.0.0
	 */
	protected LinkedList<IDuration> calculatePeriods(List<IDuration> durations){
		
		LinkedList<IDuration> periods = new LinkedList<IDuration>();
		IDuration first = durations.get(0);
		
		long periodStart = first.getStart();
		long periodEnd = first.getStart();
		
		for(IDuration d : durations){

			if(d.getEnd() <= periodEnd+idleTimeout){
				// if we're within the timeout window, add to the current period
				periodEnd = d.getEnd();
			} else {
				// beyond the timeout period, so start a new period
				periods.add(new Duration(periodStart, periodEnd));
				periodStart = d.getStart();
				periodEnd = d.getEnd();
			}
		}
		
		// add last period
		periods.add(new Duration(periodStart, periodEnd));
		return periods;
	}
	
	/**
	 * TODO: Write comment
	 *
	 * @param rep
	 * @param model
	 * @return
	 * @since nil.ucm.indications2.ui 1.0.0
	 */
	public static List<ITiming> extractTimings(IRep rep, IStoryModel model){
		
		if(rep == null) return extractAllTimings(model);
		
		List<ITiming> result = new LinkedList<ITiming>();
		
		if(!model.supports(rep))
			return result;
		
		// model metadata
		IMetaDataMap map = model.getData().getMetadata();
		if(map.containsKey(TimingMetaRep.getInstance())){
			for(IMetaDesc<ITiming> desc : map.get(TimingMetaRep.getInstance())){
				if(desc.getRep() == rep)
					result.add(desc.getData());
			}
		}
		
		// description metadat
		for(IDesc desc : model.getData().getDescriptions(rep))
			extractTimings(desc.getMetaData(), result);
		
		return result;
	}
	
	/**
	 * TODO: Write comment
	 *
	 * @param model
	 * @return
	 * @since nil.ucm.indications2.ui 1.0.0
	 */
	public static List<ITiming> extractAllTimings(IStoryModel model){
		
		List<ITiming> result = new LinkedList<ITiming>();
		
		// get model metadata
		extractTimings(model.getData().getMetadata(), result);
		
		// get desc metadata
		for(IDesc desc : model.getData())
			extractTimings(desc.getMetaData(), result);
		
		return result;
	}
	
	protected static void extractTimings(IMetaDataMap map, List<ITiming> timings){
		if(map.containsKey(TimingMetaRep.getInstance())){
			for(IMetaDesc<ITiming> desc : map.get(TimingMetaRep.getInstance())){
				timings.add(desc.getData());
			}
		}
	}
	

}
