/* Filename: IndicationCreatorPage.java
 * Creator: Raquel Hervas
 * Format: Java 2 v1.6.0
 * Date created: 30/09/2009
 */
package nil.ucm.indications2.ui.helpers;

import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import nil.ucm.indications2.core.rep.model.IIndicationStructureModel;
import nil.ucm.indications2.core.rep.model.IModifierModel;
import nil.ucm.indications2.core.rep.model.INucleusModel;
import nil.ucm.indications2.core.rep.model.IReferenceModel;
import nil.ucm.indications2.ui.content.IndicationStructureContentProvider;
import nil.ucm.indications2.ui.controllers.IIndArgController;
import nil.ucm.indications2.ui.controllers.IIndicationStructureController;
import nil.ucm.indications2.ui.controllers.IndicationStructureController;
import nil.ucm.indications2.ui.helpers.IndicationDetailsPage.ChangeIndicationTypeAction;
import nil.ucm.indications2.ui.label.IndicationStructureLabelProvider;
import nil.ucm.indications2.ui.widgets.IndArgCaptureWidget;
import nil.ucm.indications2.ui.widgets.ReferenceCaptureWidget;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.dialogs.IMessageProvider;
import org.eclipse.jface.viewers.ComboViewer;
import org.eclipse.jface.viewers.IPostSelectionProvider;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.texteditor.ITextEditorActionConstants;

import edu.mit.story.core.datamodel.ModelWrapper;
import edu.mit.story.core.notify.MessageProviderManager;
import edu.mit.story.core.validation.Message;
import edu.mit.story.ui.IStoryUIConstantIDs;
import edu.mit.story.ui.StoryUIPlugin;
import edu.mit.story.ui.actions.ActionUtils;
import edu.mit.story.ui.helpers.HelperPage;
import edu.mit.story.ui.helpers.ICreatorPage;
import edu.mit.story.ui.util.ColorManager;
import edu.mit.story.ui.widgets.ButtonAction;
import edu.mit.story.ui.widgets.MessageBar;

/** 
 *   Creator page for creating indication structures. 
 *
 * @author Raquel Hervas
 * @version 1.0, (Jan. 10, 2010)
 * @since nil.ucm.indications.ui 1.0.0
 */
public class IndicationCreatorPage extends HelperPage implements ChangeListener, ICreatorPage, IAdaptable {
	
	public static final String ID = "nil.ucm.indications2.ui.helpers.IndicationCreatorPage";

	//Messages and labels
	public static String buttonLabelCommit 		= "Commit";
	public static String buttonLabelSave 		= "Save";
	public static String buttonLabelClear 		= "Clear";
	public static String buttonLabelCancel 		= "Cancel";
	
	public static String actionLabelCommit 		= "Commit Indication";
	public static String actionLabelSave 		= "Save Indication";
	public static String actionLabelClear 		= "Clear Indication";
	public static String actionLabelCancel 		= "Cancel Edit";
	public static String actionLabelCommitRef 	= "Commit Reference";
	public static String actionLabelSaveRef 	= "Save Changes in Reference";
	public static String actionLabelCommitNuc 	= "Commit Nucleus";
	public static String actionLabelSaveNuc 	= "Save Changes in Nucleus";
	public static String actionLabelCommitMod 	= "Commit Modifier";
	public static String actionLabelSaveMod 	= "Save Changes in Modifier";
	
	public static IMessageProvider clearedMessage 	= new Message("Use this page to create a new indication structure", IMessageProvider.INFORMATION);
	public static IMessageProvider createMessage 	= new Message("Press the 'commit' button to add this indication structure to the story model", IMessageProvider.INFORMATION);
	public static IMessageProvider saveMessage 		= new Message("Press the 'save' button to change this indication structure in the story model", IMessageProvider.INFORMATION);
	
	//Controllers
	protected IIndicationStructureController controller;
	
	//View components
	protected MessageProviderManager messageManager;
	protected MessageBar messageBar;
	protected ReferenceCaptureWidget refWidget;
	protected Button toggleButton;
	protected IndArgCaptureWidget<IIndArgController> nucWidget;
	protected IndArgCaptureWidget<IIndArgController> modWidget;
	protected ComboViewer typeCombo;
	protected TreeViewer indicationViewer;
	protected Button commitNucButton;
	protected Button commitModButton;
	protected Button commitButton;
	protected Button clearButton;
	protected ButtonAction commitNucAction;
	protected ButtonAction commitModAction;
	protected ButtonAction commitAction;
	protected ButtonAction clearAction;
	protected ChangeIndicationTypeAction changeDEAction, changeDIAction, changeOTAction, changeUNAction;
	protected Label nucLabel;
	protected Label modLabel;
	/* 
	 * (non-Javadoc) @see org.eclipse.ui.part.IPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createControl(Composite parent) {
		
		Composite control = new Composite(parent, SWT.NONE);
		setControl(control);
		
		//Set up controller
		controller = new IndicationStructureController(getStoryEditor(),IStoryUIConstantIDs.ANNO_Target1,
				IStoryUIConstantIDs.ANNO_Target2, IStoryUIConstantIDs.ANNO_Target3);
		
		// the topmost composite has the message bar on top and the widget composite down below.
		//   It is also divided in two: on the left the widgets for selecting and operating with elements,
		//   and on the right a tree viewer that displays the current indication structure.
		//   The left hand has four columns (thought for two sets of label+widget).
		
		int numColumns = 1;
		GridLayout layout = new GridLayout(numColumns, false);
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		layout.horizontalSpacing = 0;
		layout.verticalSpacing = 0;
		control.setLayout(layout);
		control.setBackground(StoryUIPlugin.getDefault().getColorManager().get(ColorManager.COLOR_KEY_White));
		control.setForeground(StoryUIPlugin.getDefault().getColorManager().get(ColorManager.COLOR_KEY_Black));
		
		// at the top there is a message bar for displaying feedback to the user
		// that runs across the whole widget
		messageBar = new MessageBar(control, SWT.NONE);
		messageBar.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false, numColumns, 1));
		messageBar.setBackground(control.getBackground());
		messageBar.setForeground(control.getForeground());
		
		// separator between message bar and rest
		Label rule = new Label(control, SWT.SEPARATOR | SWT.HORIZONTAL);
		rule.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false, numColumns, 1));
		
		SashForm sash = new SashForm(control, SWT.HORIZONTAL);
		sash.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));		
		
		// lefthand side ======================================================
		// widget composite
		// the widget composite is formed by three columns
		Composite widgetParent = new Composite(sash, SWT.NONE);
		widgetParent.setLayoutData(new GridData(SWT.FILL, SWT.TOP, true, false));
		widgetParent.setBackground(control.getBackground());
		widgetParent.setForeground(control.getForeground());		
		widgetParent.setLayout(new GridLayout(3, false));

		Label label;		
		
		// first row
		// first column: reference input label
		label = new Label(widgetParent, SWT.NONE);
		label.setBackground(widgetParent.getBackground());
		label.setForeground(widgetParent.getForeground());
		label.setText("Reference");
		label.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, false, false));
		
		// second & third column:reference widget
		refWidget = new ReferenceCaptureWidget(widgetParent, SWT.NONE, controller.getReferenceController()){

			protected void createAdditionalControls(Composite parent){
				parent.setLayout(new GridLayout(2, false));
				
				Label label = new Label(parent, SWT.NONE);
				label.setText("Copular Predicate");
				label.setBackground(parent.getBackground());
				label.setForeground(parent.getForeground());
				
				toggleButton = new Button(parent, SWT.CHECK);
				toggleButton.setBackground(parent.getBackground());
				toggleButton.setForeground(parent.getForeground());
				new ButtonAction(toggleButton){
					public void run(){
						IndicationCreatorPage.this.controller.toggleCopularPredicate();
					}
				};
			}
		};
		refWidget.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false, 2, 1));
		refWidget.getCaptureAction().setText("Capture Reference");
		refWidget.getCaptureAction().setToolTipText("Capture Reference");
		refWidget.getClearAction().setText("Clear Reference");
		refWidget.getClearAction().setToolTipText("Clear Reference");		
		
		// second row
		// left side (first column): nucleus input label
		nucLabel = new Label(widgetParent, SWT.NONE);
		nucLabel.setBackground(widgetParent.getBackground());
		nucLabel.setForeground(widgetParent.getForeground());
		nucLabel.setText("Nucleus");
		nucLabel.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, false, false));		
		
		// second column: add nucleus button
		commitNucButton = new Button(widgetParent, SWT.NONE);
		commitNucButton.setText("Commit");
		commitNucAction = new ButtonAction(commitNucButton){
			public void run() {
				controller.addNucleus();
			}
		};
		commitNucAction.setText("Commit Nucleus");
		commitNucAction.setToolTipText("Commit Nucleus");
		
		// right side (third column): nucleus widget
//		nucWidget = new SegmentSetCaptureWidget<ISegmentSetController>(widgetParent, SWT.NONE, controller.getNucleusController());
		nucWidget = new IndArgCaptureWidget<IIndArgController>(widgetParent, SWT.NONE, controller.getNucleusController());
		nucWidget.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
		nucWidget.getAddAction().setText("Add to Nucleus");
		nucWidget.getAddAction().setToolTipText("Add to Nucleus");
		nucWidget.getSubtractAction().setText("Subtract from Nucleus");
		nucWidget.getSubtractAction().setToolTipText("Subtract from Nucleus");
		nucWidget.getClearAction().setText("Clear Nucleus");
		nucWidget.getClearAction().setToolTipText("Clear Nucleus");		
		
		
		//third row		
		// left side (first column): modifier input label
		modLabel = new Label(widgetParent, SWT.NONE);
		modLabel.setBackground(widgetParent.getBackground());
		modLabel.setForeground(widgetParent.getForeground());
		modLabel.setText("Modifier");
		modLabel.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, false, false));
		
		// second column: add modifier button
		commitModButton = new Button(widgetParent, SWT.NONE);
		commitModButton.setText("Commit");
		commitModAction = new ButtonAction(commitModButton){
			public void run() {
				controller.addModifier();
			}
		};
		commitModAction.setText("Commit Modifier");
		commitModAction.setToolTipText("Commit Modifier");
		
		// third column: modifier widget
//		modWidget = new SegmentSetCaptureWidget<ISegmentSetController>(widgetParent, SWT.NONE, controller.getModifierController());
		modWidget = new IndArgCaptureWidget<IIndArgController>(widgetParent, SWT.NONE, controller.getModifierController());
		modWidget.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
		modWidget.getAddAction().setText("Add to Modifier");
		modWidget.getAddAction().setToolTipText("Add to Modifier");
		modWidget.getSubtractAction().setText("Subtract from Modifier");
		modWidget.getSubtractAction().setToolTipText("Subtract from Modifier");
		modWidget.getClearAction().setText("Clear Modifier");
		modWidget.getClearAction().setToolTipText("Clear Modifier");
		

		// fourth row
		// indication structure lower buttons
		Composite lowerButtons = new Composite(widgetParent, SWT.NONE);
		lowerButtons.setBackground(widgetParent.getBackground());
		lowerButtons.setForeground(widgetParent.getForeground());
		GridData grid = new GridData(SWT.RIGHT, SWT.CENTER, false, false);
		grid.horizontalSpan = 3;
		lowerButtons.setLayoutData(grid);
		layout = new GridLayout(2, false);
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		lowerButtons.setLayout(layout);
		
		
		// commit button
		commitButton = new Button(lowerButtons, SWT.NONE);
		commitButton.setText("Commit");
		commitAction = new ButtonAction(commitButton){
			public void run() {
				controller.commit();
			}
		};
		
		// clear button
		clearButton = new Button(lowerButtons, SWT.NONE);
		clearButton.setText("Clear");
		clearAction = new ButtonAction(clearButton){
			public void run() {
				controller.clear();
			}
		};
		
		// righthand side =====================================================
		// a tree viewer that displays the current reference
		indicationViewer = new TreeViewer(sash, SWT.BORDER);
		indicationViewer.setContentProvider(new IndicationStructureContentProvider(getStoryEditor()));
        indicationViewer.setLabelProvider(new IndicationStructureLabelProvider(getStoryEditor()));
        indicationViewer.setInput(new ModelWrapper(controller.getIndicationStructureModel()));
        indicationViewer.addSelectionChangedListener(controller.getActionEdit());
        indicationViewer.addSelectionChangedListener(controller.getActionDelete());
		
		// set weights on the sash
		sash.setWeights(new int[]{3, 1});
		
		// create popup menu manager
		final MenuManager manager = new MenuManager();
		ActionUtils.createPopupGroups(manager);
		manager.appendToGroup(ITextEditorActionConstants.GROUP_EDIT, controller.getActionEdit());
		manager.appendToGroup(ITextEditorActionConstants.GROUP_EDIT, controller.getActionDelete());
		
		// create & register popup menu
		Control tree = indicationViewer.getTree();
		Menu menu = manager.createContextMenu(tree);
		tree.setMenu(menu);
		
		//Set up message manager
		messageManager = new MessageProviderManager(this);
		messageManager.add(controller.getIndicationStructureModel());
		messageManager.add(controller.getNucleusController().getModel(), true);
		messageManager.add(controller.getModifierController().getModel(), true);
		messageManager.addChangeListener(this);
		
		// messages
		messageManager.setAllClearedMesage(clearedMessage);
		
		updateFromModel();
	}
	
	/** Syncs the view with the model */
	protected void updateFromModel(){

		//Get models
		IIndicationStructureModel model = controller.getIndicationStructureModel();
		IReferenceModel refModel = controller.getReferenceController().getModel();
		INucleusModel nucModel = controller.getNucleusController().getModel();
		IModifierModel modModel = controller.getModifierController().getModel();
		
		// loaded?
		boolean isLoaded = model.getLoaded() != null;
		
		// messages
		IMessageProvider defaultMsg = isLoaded ? saveMessage : createMessage;
		if(!messageManager.getDefaultMessage().equals(defaultMsg)) messageManager.setDefaultMesage(defaultMsg);
		messageBar.setMessage(messageManager.getMessage(), messageManager.getMessageType());
		
		
		// calculate expected state
		boolean enableAddNuc = !nucModel.isCleared();
		boolean enableAddMod = !modModel.isCleared();
		boolean enableCommit = !model.isCleared() && messageManager.getMessageType() != IMessageProvider.ERROR; 
		boolean enableClear = !model.isCleared() || !refModel.isCleared() || !nucModel.isCleared() || !modModel.isCleared();
		boolean isCop = controller.getIndicationStructureModel().isCopularPredicate();
		
		String commitButtonLabel = isLoaded ? buttonLabelSave : buttonLabelCommit;
		String clearButtonLabel = isLoaded ? buttonLabelCancel : buttonLabelClear;
		
		String commitActionLabel = isLoaded ? actionLabelSave : actionLabelCommit;
		String clearActionLabel = isLoaded ? actionLabelCancel : actionLabelClear;
		
		// update view state
		if(enableAddNuc != commitNucButton.isEnabled()) commitNucButton.setEnabled(enableAddNuc);
		if(enableAddMod != commitModButton.isEnabled()) commitModButton.setEnabled(enableAddMod);
		if(enableCommit != commitButton.isEnabled()) commitButton.setEnabled(enableCommit);
		if(enableClear != clearButton.isEnabled()) clearButton.setEnabled(enableClear);
		if(toggleButton.getSelection() != isCop) toggleButton.setSelection(isCop);
		
		// update action labels
		if(!commitActionLabel.equals(commitAction.getText())){
			commitAction.setText(commitActionLabel);
			commitAction.setToolTipText(commitActionLabel);
		}
		if(!clearActionLabel.equals(clearAction.getText())){
			clearAction.setText(clearActionLabel);
			clearAction.setToolTipText(clearActionLabel);
		}
		
		// update button labels
		boolean labelChanged = false;
		if(!commitButtonLabel.equals(commitButton.getText())){
			labelChanged = true;
			commitButton.setText(commitButtonLabel);
		}
		if(!clearButtonLabel.equals(clearButton.getText())){
			labelChanged = true;
			clearButton.setText(clearButtonLabel);
		}
		
		// refresh tree
		indicationViewer.getTree().setRedraw(false);
		indicationViewer.refresh();
		indicationViewer.expandAll();
		indicationViewer.getTree().setRedraw(true);
		
		// layout if needed
		if(labelChanged) 
			((Composite)getControl()).layout(true, true);
		
	}
	
	/* 
	 * (non-Javadoc) @see javax.swing.event.ChangeListener#stateChanged(javax.swing.event.ChangeEvent)
	 */
	public void stateChanged(ChangeEvent arg0) {
		updateFromModel();
	}

	public IIndicationStructureController getController(){
		return controller;
	}

	/* 
	 * (non-Javadoc) @see org.eclipse.ui.part.IPage#setActionBars(org.eclipse.ui.IActionBars)
	 */
	
	public void setActionBars(IActionBars actionBars) {
	
		// delete
		actionBars.setGlobalActionHandler(ActionFactory.DELETE.getId(), getController().getActionDelete());
		
		// add
		refWidget.getCaptureAction().setActionDefinitionId(IStoryUIConstantIDs.COMMAND_AddToTarget1);
		nucWidget.getAddAction().setActionDefinitionId(IStoryUIConstantIDs.COMMAND_AddToTarget2);
		modWidget.getAddAction().setActionDefinitionId(IStoryUIConstantIDs.COMMAND_AddToTarget3);
		actionBars.setGlobalActionHandler(refWidget.getCaptureAction().getActionDefinitionId(), refWidget.getCaptureAction());
		actionBars.setGlobalActionHandler(nucWidget.getAddAction().getActionDefinitionId(), nucWidget.getAddAction());
		actionBars.setGlobalActionHandler(modWidget.getAddAction().getActionDefinitionId(), modWidget.getAddAction());
	
		// subtract
		nucWidget.getSubtractAction().setActionDefinitionId(IStoryUIConstantIDs.COMMAND_SubtractFromTarget2);
		modWidget.getSubtractAction().setActionDefinitionId(IStoryUIConstantIDs.COMMAND_SubtractFromTarget3);
		actionBars.setGlobalActionHandler(nucWidget.getSubtractAction().getActionDefinitionId(), nucWidget.getSubtractAction());
		actionBars.setGlobalActionHandler(modWidget.getSubtractAction().getActionDefinitionId(), modWidget.getSubtractAction());
	
		// commit
		commitAction.setActionDefinitionId(IStoryUIConstantIDs.COMMAND_CommitTarget0);
		commitNucAction.setActionDefinitionId(IStoryUIConstantIDs.COMMAND_CommitTarget2);
		commitModAction.setActionDefinitionId(IStoryUIConstantIDs.COMMAND_CommitTarget3);
		actionBars.setGlobalActionHandler(commitAction.getActionDefinitionId(), commitAction);
		actionBars.setGlobalActionHandler(commitNucAction.getActionDefinitionId(), commitNucAction);
		actionBars.setGlobalActionHandler(commitModAction.getActionDefinitionId(), commitModAction);
		
		// clear
		clearAction.setActionDefinitionId(IStoryUIConstantIDs.COMMAND_ClearTarget0);
		refWidget.getClearAction().setActionDefinitionId(IStoryUIConstantIDs.COMMAND_ClearTarget1);
		nucWidget.getClearAction().setActionDefinitionId(IStoryUIConstantIDs.COMMAND_ClearTarget2);
		modWidget.getClearAction().setActionDefinitionId(IStoryUIConstantIDs.COMMAND_ClearTarget3);
		actionBars.setGlobalActionHandler(clearAction.getActionDefinitionId(), clearAction);
		actionBars.setGlobalActionHandler(refWidget.getClearAction().getActionDefinitionId(), refWidget.getClearAction());
		actionBars.setGlobalActionHandler(nucWidget.getClearAction().getActionDefinitionId(), nucWidget.getClearAction());
		actionBars.setGlobalActionHandler(modWidget.getClearAction().getActionDefinitionId(), modWidget.getClearAction());
		
	}
	
	/* 
	 * (non-Javadoc) @see org.eclipse.core.runtime.IAdaptable#getAdapter(java.lang.Class)
	 */
	
	@SuppressWarnings("unchecked")
	public Object getAdapter(Class adapter) {
		// return requests for selection providers
		if(adapter == ISelectionProvider.class) return indicationViewer;
		if(adapter == IPostSelectionProvider.class) return indicationViewer;
		// otherwise defer to platform's adapter manager
		return Platform.getAdapterManager().getAdapter(this, adapter);
	}

}


